<!--  ********************************************************************* -->
<!--  XSLT Transform to read UCD Create display styles XML
      and output the JavaScript needed to create the corresponding
      CSS in the calling DHtml.
-->
<!--  ********************************************************************* -->

<xsl:stylesheet version="1.0"
     xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
     xmlns:msxsl="urn:schemas-microsoft-com:xslt"
     xmlns:user="urn:my-scripts"
     xmlns:const="urn:my-constants"
>
<xsl:output method="text"/>

<!--  ********************************************************************* -->
<!--  const:MAPPING when @style is encountered, create rules for these
      named CSS classes, which will may be used as constants -->
<const:MAPPING style="title">.ds_TitleStyle</const:MAPPING>
<const:MAPPING style="_buttonctrl">.ds_ButtonStyle</const:MAPPING>

<!--  const:style use these text values for CSS class names -->
<const:style   name="prefix">.ds_style</const:style>
<const:style   name="TextStyle">.ds_TextStyle</const:style>
<const:style   name="MainStyle">.ds_MainStyle</const:style>
<const:style   name="ValueStyle">.ds_ValueStyle</const:style>
<const:style   name="HyperStyle">.ds_HyperStyle</const:style>
<const:style   name="body">body</const:style>

<!-- const:rule additional rules for style table entries we find -->
<const:rule    name=".ds_ButtonStyle" rule="text-decoration">underline;</const:rule>

<!-- const:engine rules we will always create -->
<const:engine   name="HyperStyle">color:blue;text-decoration:underline;</const:engine>

<!-- ********************************************************************* -->
<!-- **************************  MAIN   ********************************** -->
<!-- ********************************************************************* -->

<!--  ********************************************************************* -->
<!--  XSLT Main; matches root node and gets everything started -->
<xsl:template match="/">
   <xsl:text>
   ss = document.createStyleSheet();
   </xsl:text>
   <xsl:apply-templates select="document('')/xsl:stylesheet/const:engine"/>
   <xsl:apply-templates select="/*/Style"/>
</xsl:template>

<!--  ********************************************************************* -->
<!--  Template to create JavaScript for a constant rules
-->
<xsl:template match="const:rule">
   <xsl:value-of select="@rule"/><xsl:text>:</xsl:text><xsl:value-of select="text()"/>
</xsl:template>

<xsl:template match="const:engine">
   <xsl:param name="commonName" select="@name"/>
   <xsl:param name="ruleName" select="document('')/xsl:stylesheet/const:style[@name=$commonName]"/>
   <xsl:param name="ruleValue" select="text()"/>
   <xsl:text>ss.addRule("</xsl:text><xsl:value-of select="$ruleName"/><xsl:text>","</xsl:text><xsl:value-of select="$ruleValue"/><xsl:text>");
   </xsl:text>
</xsl:template>


<!--  ********************************************************************* -->
<!--  Template to create JavaScript for a specific CSS rule
-->
<xsl:template name="addRule">
   <xsl:param name="ruleName"/>
   <xsl:param name="ruleValue"/>
   <xsl:text>ss.addRule("</xsl:text><xsl:value-of select="$ruleName"/><xsl:text>","</xsl:text><xsl:value-of select="$ruleValue"/><xsl:text>");
   </xsl:text>
</xsl:template>

<!--  ********************************************************************* -->
<!--  Template to create all CSS rules for a given Style tag
-->
<xsl:template name="createStyle">
   <!-- The name of the CSS class or element we are creating rules for -->
   <xsl:param name="styleName"/>
   <xsl:param name="commonName"/>
   
   <!--  put the following attributes in variables so they can be used
         in XPath queries -->
   <xsl:param name="font" select="@font"/>
   <xsl:param name="color" select="@color"/>
   <xsl:param name="highlight" select="@highlight"/>
   <xsl:param name="bold" select="@bold"/>
   <xsl:param name="italic" select="@italic"/>

   <!-- Build rules for style -->
   <xsl:param name="allRules">
      <xsl:if test="$font">
         <xsl:text>font: </xsl:text>
         <!-- convert to lower case before comparison -->
         <xsl:if test="translate(string($bold),'TRUE','true')='true'">
            <xsl:text>BOLD </xsl:text>
         </xsl:if>
         <!-- convert to lower case before comparison -->
         <xsl:if test="translate(string($italic),'TRUE','true')='true'">
            <xsl:text>ITALIC </xsl:text>
         </xsl:if>
         <!-- Now pull out font size and name from FontSpec branch -->
         <xsl:apply-templates select="/*/FontSpec[@name=$font]"/>
         <xsl:text>;</xsl:text>
      </xsl:if>
      <xsl:if test="$color">
         <xsl:text>color: </xsl:text>
         <!-- Pull out color rgb value from ColorSpec branch -->
         <xsl:apply-templates select="/*/ColorSpec[@name=$color]"/>
         <xsl:text>;</xsl:text>
      </xsl:if>
      <xsl:if test="$highlight">
         <xsl:text>background-color: </xsl:text>
         <!-- Pull out color rgb value from ColorSpec branch -->
         <xsl:apply-templates select="/*/ColorSpec[@name=$highlight]"/>
         <xsl:text>;</xsl:text>
      </xsl:if>
      <xsl:if test="$commonName">
         <xsl:apply-templates select="document('')/xsl:stylesheet/const:rule[@name=$commonName]"/>
      </xsl:if>
   </xsl:param>

   <!-- Create Rule JavaScript -->
   <xsl:call-template name="addRule">
      <xsl:with-param name="ruleName" select="$styleName"/>
      <xsl:with-param name="ruleValue" select="$allRules"/>
   </xsl:call-template>

</xsl:template>

<!--  ********************************************************************* -->
<!--  Tempalte for FontSpec tag; diverts to child Font tag
-->
<xsl:template match="FontSpec">
   <xsl:apply-templates select="Font"/>
</xsl:template>

<!--  ********************************************************************* -->
<!--  Template for Font tag;  Need to find Font tag that applies to this
      OS and use it as specified by cond attribute.  If it contains 'win' 
      or 'any', then this is the right Font tag, and it will output rule
      directives
-->
<xsl:template match="Font">
   <!-- convert cond attribute to lower case for case insesitive comparison -->
   <xsl:if test="contains(translate(string(@cond),'WINANY','winany'),'win') or contains(translate(string(@cond),'WINANY','winany'),'any')">
      <xsl:value-of select="user:calcfontpixels(string(@FontSize))"></xsl:value-of><xsl:text>px </xsl:text> <xsl:value-of select="@FontName"></xsl:value-of>
   </xsl:if>
</xsl:template>

<!--  ********************************************************************* -->
<!--  Template for ColorSpec tag; diverts to child Color tag
-->
<xsl:template match="ColorSpec">
   <xsl:apply-templates select="Color"/>
</xsl:template>

<!--  ********************************************************************* -->
<!--  Template for Color tag;  Need to find Color tag that applies to this
      OS and use it as specified by cond attribute.  If it contains 'win' 
      or 'any', then this is the right Font tag, and it will output rule
      directives
-->
<xsl:template match="Color">
   <!-- convert cond attribute to lower case for case insesitive comparison -->
   <xsl:if test="contains(translate(string(@cond),'WINANY','winany'),'win') or contains(translate(string(@cond),'WINANY','winany'),'any')">
      <xsl:value-of select="@rgb"/>
   </xsl:if>
</xsl:template>

<!--  ********************************************************************* -->
<!--  This is the template that is matched by teh Main "/" template which
      gets the whole thing started.  For each Style, it may create a number
      of CSS clases based on the Style tags attributes.  In addition, it
      will create a body element when it finds the "MainStyle".  
-->
<xsl:template match="Style">
   <xsl:param name="styleName"/>
   
   <!-- build name for named styles -->
   <xsl:param name="named_styleName">
      <xsl:value-of select="document('')/xsl:stylesheet/const:style[@name='prefix']"/>
      <xsl:value-of select="@name"></xsl:value-of>
   </xsl:param>

   <!-- build name for indexed styles -->
   <xsl:param name="indexed_styleName">
      <xsl:value-of select="document('')/xsl:stylesheet/const:style[@name='prefix']"/>
      <xsl:value-of select="@index"></xsl:value-of>
   </xsl:param>

   <!--  put name attriube in variable, so it can be referenced as $name in 
         XPath queries, ie. const:MAPPING[@style=$name] -->
   <xsl:param name="name" select="@name"/>

   <xsl:if test="@name">
      <!--  The name attribute is specified, create rules for named styles -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="$named_styleName"/>
      </xsl:call-template>

      <!--  Lookup this style name in the const:MAPPING table, and create
            corresponding speicial CSS class if found -->
      <xsl:if test="document('')/xsl:stylesheet/const:MAPPING[@style=$name]">
         <xsl:call-template name="createStyle">
            <xsl:with-param name="styleName" select="document('')/xsl:stylesheet/const:MAPPING[@style=$name]/text()"/>
            <xsl:with-param name="commonName" select="document('')/xsl:stylesheet/const:MAPPING[@style=$name]/text()"/>
         </xsl:call-template>
      </xsl:if>
   </xsl:if>
  
   <xsl:if test="@index">
      <!-- The index attribute is specified, create rules for indexed styles -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="$indexed_styleName"/>
      </xsl:call-template>
   </xsl:if>

   <!-- Convert to lower case, then compare -->   
   <xsl:if test="translate(string(@MainStyle),'TRUE','true')='true'">
      <!-- Create the special body element, using this Style -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="document('')/xsl:stylesheet/const:style[@name='body']/text()"/>
         <xsl:with-param name="commonName" select="document('')/xsl:stylesheet/const:MAPPING[@style='body']/text()"/>
      </xsl:call-template>

      <!-- Create the speical Main Style, using this Style -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="document('')/xsl:stylesheet/const:style[@name='MainStyle']/text()"/>
         <xsl:with-param name="commonName" select="document('')/xsl:stylesheet/const:MAPPING[@style='MainStyle']/text()"/>
      </xsl:call-template>
   </xsl:if>

   <!-- Convert to lower case, then compare -->   
   <xsl:if test="translate(string(@TextStyle),'TRUE','true')='true'">
      <!-- Create the speical Text Style, using this Style -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="document('')/xsl:stylesheet/const:style[@name='TextStyle']/text()"/>
         <xsl:with-param name="commonName" select="document('')/xsl:stylesheet/const:MAPPING[@style='TextStyle']/text()"/>
      </xsl:call-template>
   </xsl:if>

   <!-- Convert to lower case, then compare -->   
   <xsl:if test="translate(string(@ValueStyle),'TRUE','true')='true'">
      <!-- Create the speical Value Style, using this Style -->
      <xsl:call-template name="createStyle">
         <xsl:with-param name="styleName" select="document('')/xsl:stylesheet/const:style[@name='ValueStyle']/text()"/>
         <xsl:with-param name="commonName" select="document('')/xsl:stylesheet/const:MAPPING[@style='ValueStyle']/text()"/>
      </xsl:call-template>
   </xsl:if>

</xsl:template>

<!-- ******************************************************************************** -->
<!-- ******************************************************************************** -->
<!-- *************************  JavaScript functions ******************************** -->
<!-- ******************************************************************************** -->
<!-- ******************************************************************************** -->

<msxsl:script language="JavaScript" implements-prefix="user">
<![CDATA[
        // return pixel size at 96DPI to ensure consitent display reguardless DPI setting
        function calcfontpixels(pointSize)
        {
           return (pointSize*96)/72;
        }
]]>
</msxsl:script>

</xsl:stylesheet>




